////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//  PDF.h
//      Definition of the PDF class. Class is the encapsulation of a PDF.
//
//  Copyright (C) 2010 MYOB US, Inc., all rights reserved.
//
//  Change History (most recent first):
//
//  08/29/10  K.L.S.  Created file.
//
//  Notes:
//      To use this class the minimium you need to do is the following:
//          PDF pdf;
//          pdf.BeginDocument(PaperRect, PageRect, IsLandscaped); // you can set the page and paper size and orientation if you choose
//      ---- Do the following lines for as many pages as you want
//          pdf.BeginPage();                                      // Starts a new page for output
//          // draw your page
//          pdf.Draw();                                           // Several forms of draw (text, picures, lines, rectangles)
//          pdf.EndPage();
//      ---- End for pages
//          pdf.EndDocument();                                    // Ends the document
//          pdf.Output(OutputFilePath);                           // Full path and name of the file you want to create
//
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma once

#include "CrossPlatform/CrossPlatform.h"
#include "ObsoleteBridge/OBFontSpec.h"
#include "ObsoleteBridge/OBColor.h"
#include "ObsoleteBridge/OBImage.h"
#include "ObsoleteBridge/PDFPolygonInfo.h"

namespace ObsoleteBridge
{
	namespace PDFSupport
	{
		class EXTERNAL_API PDF
		{
		public:
			static PDF* Create(void);

			virtual ~PDF(void);
			
			virtual bool ShowPrintDialog(void) = 0;

			virtual CPError BeginDocument(void) = 0;
			virtual CPError BeginDocument(const CPRect& PageFrame) = 0;
			virtual CPError BeginDocument(const CPRect& PageFrame, bool IsLandscape) = 0;
			virtual CPError BeginDocument(const CPRect& PageFrame, const CPRect& PrintableAreaFrame, bool IsLandscape) = 0;
			virtual CPError EndDocument(void) = 0;
			
			virtual CPError BeginPage(void) = 0;
			virtual CPError EndPage(void) = 0;

			virtual CPError Draw(const CPRect& Frame, const OBColor& FrameColor, const OBColor* FillColor, CPCoordinate LineWidth) = 0;
			virtual CPError Draw(const OBImage& Image, const CPRect& Frame) = 0;
			virtual CPError Draw(const OBImage& Image, const CPRect& Frame, const CPRect& ClipRect) = 0;
			virtual CPError Draw(const CPString& Text, const CPRect& Frame, const OBFontSpec& Font, const OBColor& Color) = 0;
			virtual CPError Draw(const PDFPolygonInfo& PolygonPoints, const OBColor& FrameColor, const OBColor* FillColor, CPCoordinate LineWidth) = 0;
			virtual CPError DrawOval(const CPRect& Frame, const OBColor& FrameColor, const OBColor* FillColor, CPCoordinate LineWidth) = 0;
			virtual CPError DrawRoundedRectangle(const CPRect&Frame, const OBColor& FrameColor, const OBColor* FillColor, CPCoordinate LineWidth) = 0;

			virtual CPError SetClipRect(const CPRect& ClipRect) = 0;

			virtual CPCoordinate GetLineHeight(void) const = 0;
			virtual CPCoordinate GetLineHeight(const OBFontSpec&) const = 0;
			virtual CPCoordinate GetFontHeight(void) const = 0;
			virtual CPCoordinate GetFontHeight(const OBFontSpec&) const = 0;
			virtual CPCoordinate GetFontHeightWithoutSpacing(void) const = 0;
			virtual CPCoordinate GetFontHeightWithoutSpacing(const OBFontSpec&) const = 0;

			virtual CPCoordinate GetStringWidth(const CPString&) const = 0;
			virtual CPCoordinate GetStringWidth(const CPString&, const OBFontSpec&) = 0;
	
			virtual CPError Output(const CPString&) = 0;
			virtual CPError Print(bool ToPreview) = 0;

			virtual void SetDashLine(bool) = 0;
			virtual void SetFont(const OBFontSpec&) = 0;

			virtual void StartBatchPrinting(void) = 0;
			virtual void EndBatchPrinting(void) = 0;
			virtual bool IsBatchPrinting(void) const = 0;
			virtual void SetScalingDisabled(bool DisableScaling) = 0;

			//=========================
			//GRW's methods:
			virtual void SetInternalTitle(const CPString& TitleString) = 0;

		protected:
			PDF(void);
		};
	}
}
